//
//  TWStreamsCollectionViewCell.m
//  Xarth
//
//  Created by Auston Stewart on 10/22/13.
//  Copyright (c) 2013 Justin.tv, Inc. All rights reserved.
//

#import "TWStreamsCollectionViewCell.h"
#import "UIFont+Twitch.h"
#import "NSString+Twitch.h"

NSString *const TWStreamsCollectionViewCellReuseIdentifier = @"TWStreamsCollectionViewCellReuseIdentifier";

@implementation TWStreamsCollectionViewCell
{
	UILabel *_channelNameLabel;
	UILabel *_detailTextLabel;
	UILabel *_broadcastTitleLabel;
	
	CALayer *_enlargedStreamGradientOverlay;
	UIImage *_portraitGradientOverlay;
	UIImage *_landscapeGradientOverlay;
	
	BOOL _viewerCountIsSet;
}

#define TWStreamCellTextPadding 10.f
#define TWStreamCellBroadcasterLabelBottomOffsetWithTitle 61.f
#define TWStreamCellBroadcasterLabelBottomOffsetWithoutTitle 43.f
#define TWStreamCellBroadcasterLabelHeight 19.f
#define TWStreamCellBroadcastTitleBottomOffset 39.f
#define TWStreamCellBroadcastTitleLabelHeight 14.f
#define TWStreamCellViewerCountLabelBottomOffset 22.f
#define TWStreamCellViewerCountLabelHeight 14.f
	
+ (CGSize)portraitCellSize
{
	return CGSizeMake(369.f,208.f);
}
	
+ (CGSize)landscapeCellSize
{
	return CGSizeMake(328.f,184.f);
}

- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
		
		NSString *placeholderImageName = nil;
		if (CGSizeEqualToSize([[self class] portraitCellSize], self.frame.size))
			placeholderImageName = @"404_channel_ipad_grid";
		else placeholderImageName = @"404_sdk_ipad_landscape_directory";
		
		if (placeholderImageName) self.placeholderImageName = placeholderImageName;
		
		_enlargedStreamGradientOverlay = [[CALayer alloc] init];
		_portraitGradientOverlay = [UIImage imageFromFrameworkBundleNamed:@"gradient_sdk_ipad_portrait_directory"];
		_landscapeGradientOverlay = [UIImage imageFromFrameworkBundleNamed:@"gradient_sdk_ipad_landscape_directory"];
		self.backgroundView.layer.masksToBounds = YES;
		[self.backgroundView.layer addSublayer:_enlargedStreamGradientOverlay];
		
		// Create custom labels
		_channelNameLabel = [[UILabel alloc] initWithFrame:CGRectMake(TWStreamCellTextPadding, frame.size.height - TWStreamCellBroadcasterLabelBottomOffsetWithoutTitle, frame.size.width - (2.f * TWStreamCellTextPadding), TWStreamCellBroadcasterLabelHeight)];
		_channelNameLabel.font = [UIFont twitchEnlargedLiveChannelCellBroadcasterFont];
		_channelNameLabel.textColor = [UIColor twitchWhiteOverlayTextColor];
		_channelNameLabel.autoresizingMask = UIViewAutoresizingFlexibleTopMargin;
		
		_detailTextLabel = [[UILabel alloc] initWithFrame:CGRectMake(TWStreamCellTextPadding, frame.size.height - TWStreamCellViewerCountLabelBottomOffset, frame.size.width - (2.f * TWStreamCellTextPadding), TWStreamCellViewerCountLabelHeight)];
		_detailTextLabel.font = [UIFont twitchEnlargedLiveChannelCellViewerCountFont];
		_detailTextLabel.textColor = [UIColor twitchPurpleOverlayTextColor];
		_detailTextLabel.autoresizingMask = UIViewAutoresizingFlexibleTopMargin;
		
		// Broadcast titles must clear the right 1/3rd of the cell
		_broadcastTitleLabel = [[UILabel alloc] initWithFrame:CGRectMake(TWStreamCellTextPadding, frame.size.height - TWStreamCellBroadcastTitleBottomOffset, frame.size.width - TWStreamCellTextPadding - floorf(frame.size.width / 3.f), TWStreamCellBroadcastTitleLabelHeight)];
		_broadcastTitleLabel.font = [UIFont twitchCellBroadcastTitleFont];
		_broadcastTitleLabel.textColor = [UIColor twitchGridItemBroadcastTitleColor];
		_broadcastTitleLabel.autoresizingMask = UIViewAutoresizingFlexibleTopMargin | UIViewAutoresizingFlexibleWidth;
		_broadcastTitleLabel.textAlignment = NSTextAlignmentLeft;
		_broadcastTitleLabel.lineBreakMode = NSLineBreakByTruncatingTail;
		_broadcastTitleLabel.hidden = YES;
		
		/*
		 TWGradientOverlayView *gradientOverlay = [[TWGradientOverlayView alloc] initWithFrame:CGRectMake(0.f, 0.f, frame.size.width, 68.f) startingOpacity:0.8f];
		 gradientOverlay.direction = TWGradientDirectionDown;
		 
		 [self.contentView addSubview:gradientOverlay];
		 */
		
		[self.contentView addSubview:_broadcastTitleLabel];
		[self.contentView addSubview:_channelNameLabel];
		[self.contentView addSubview:_detailTextLabel];
    }
    return self;
}

- (void)setChannelDisplayName:(NSString *)channelDisplayName
{
	_channelDisplayName = [channelDisplayName copy];
	_channelNameLabel.text = channelDisplayName;
}

- (void)prepareForReuse
{
	[super prepareForReuse];
	
	_viewerCountIsSet = NO;
	_gameName = nil;
	
	// Hide the broadcast title
	CGRect rect = _channelNameLabel.frame;
	rect.origin.y = self.contentView.frame.size.height - TWStreamCellBroadcasterLabelBottomOffsetWithoutTitle;
	_channelNameLabel.frame = rect;
	_broadcastTitleLabel.hidden = YES;
}

- (void)updateCombinedDetailTextLabel
{
	NSString *viewersString = [@(_viewerCount) formattedLocalizedNumericRepresentation];
	NSString *localizedCombinedDetailTextString = TKLocalizedString(@"playing %@ for %@ viewers",@"Enlarged stream cell combined detail text");
	NSRange gameNameSpecifierRange = [localizedCombinedDetailTextString tw_rangeOfStringSpecifier:1];
	NSRange viewerCountSpecifierRange = [localizedCombinedDetailTextString tw_rangeOfStringSpecifier:2];
	
	NSMutableAttributedString *attributedDetailText = [[NSMutableAttributedString alloc] initWithString:[NSString		stringWithFormat:TKLocalizedString(@"playing %@ for %@ viewers",@"Enlarged stream cell combined detail text"),_gameName,viewersString]];
	
	if (gameNameSpecifierRange.location != NSNotFound && viewerCountSpecifierRange.location != NSNotFound) {
		
		if (gameNameSpecifierRange.location > viewerCountSpecifierRange.location) {
			
			gameNameSpecifierRange.location += viewersString.length - gameNameSpecifierRange.length;
		}
		else {
			
			viewerCountSpecifierRange.location += _gameName.length - gameNameSpecifierRange.length;
		}
		
		[attributedDetailText addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(gameNameSpecifierRange.location,_gameName.length)];
		[attributedDetailText addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(viewerCountSpecifierRange.location,viewersString.length)];
	}
	
	_detailTextLabel.attributedText = attributedDetailText;
}

- (void)setViewerCount:(NSInteger)viewerCount
{
	_viewerCount = viewerCount;
	_viewerCountIsSet = YES;
	
	if (_gameName) [self updateCombinedDetailTextLabel];
	else {
		
		NSString *viewersString = [@(_viewerCount) formattedLocalizedNumericRepresentation];
		NSString *localizedViewerCountString = TKLocalizedString(@"%@ viewers", @"Enlarged stream cell viewer count text");
		NSRange viewCountSpecifierRange = [localizedViewerCountString tw_rangeOfStringSpecifier:1];
		NSMutableAttributedString *attributedViewerCount = [[NSMutableAttributedString alloc] initWithString:[NSString stringWithFormat:TKLocalizedString(@"%@ viewers", @"Enlarged stream cell viewer count text"),viewersString]];
		
		if (viewCountSpecifierRange.location != NSNotFound) {
			[attributedViewerCount addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(viewCountSpecifierRange.location,viewersString.length)];
		}
		_detailTextLabel.attributedText = attributedViewerCount;
	}
}

- (void)setGameName:(NSString *)gameName
{
	_gameName = [gameName copy];
	
	if (_viewerCountIsSet) [self updateCombinedDetailTextLabel];
	else {
		
		NSString *localizedGameNameString = TKLocalizedString(@"playing %@",@"Enlarged stream cell game name text");
		NSRange gameNameSpecifierRange = [localizedGameNameString tw_rangeOfStringSpecifier:1];
		NSMutableAttributedString *attributedGameName = [[NSMutableAttributedString alloc] initWithString:[NSString		stringWithFormat:TKLocalizedString(@"playing %@",@"Enlarged stream cell game name text"),_gameName]];
		
		if (gameNameSpecifierRange.location != NSNotFound) {
			[attributedGameName addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(gameNameSpecifierRange.location,_gameName.length)];
		}
		_detailTextLabel.attributedText = attributedGameName;
	}
}

- (void)setBroadcastTitle:(NSString *)broadcastTitle
{
	if (broadcastTitle.length) {
		
		_broadcastTitle = [broadcastTitle copy];
		_broadcastTitleLabel.text = _broadcastTitle;
		
		// Make space for the broadcast title
		CGRect rect = _channelNameLabel.frame;
		rect.origin.y = self.contentView.frame.size.height - TWStreamCellBroadcasterLabelBottomOffsetWithTitle;
		_channelNameLabel.frame = rect;
		_broadcastTitleLabel.hidden = NO;
	}
}

- (void)adjustForHighlightState:(BOOL)highlighted
{
	CGPoint broadcasterLabelOrigin = CGPointMake(TWStreamCellTextPadding, self.bounds.size.height - (_broadcastTitleLabel.hidden ? TWStreamCellBroadcasterLabelBottomOffsetWithoutTitle : TWStreamCellBroadcasterLabelBottomOffsetWithTitle));
	CGPoint broadcastTitleOrigin = CGPointMake(TWStreamCellTextPadding, self.bounds.size.height - TWStreamCellBroadcastTitleBottomOffset);
	CGPoint viewerCountLabelOrigin = CGPointMake(TWStreamCellTextPadding, self.bounds.size.height - TWStreamCellViewerCountLabelBottomOffset);
	CGRect broadcasterLabelFrame = _channelNameLabel.frame;
	CGRect broadcastTitleLabelFrame = _broadcastTitleLabel.frame;
	CGRect viewerCountLabelFrame = _detailTextLabel.frame;
	broadcasterLabelFrame.origin = broadcasterLabelOrigin;
	broadcastTitleLabelFrame.origin = broadcastTitleOrigin;
	viewerCountLabelFrame.origin = viewerCountLabelOrigin;
	CGRect bounds = self.bounds;
	bounds.origin = CGPointZero;
	
	if (highlighted) {
		
		CGSize initialSize = bounds.size;
		
		// Determine new image bounds
		bounds.size.width = floorf(.95 * bounds.size.width);
		bounds.size.height = floorf(.95 * bounds.size.height);
		
		// Determine new text frames
		CGFloat xDelta = floorf((initialSize.width - bounds.size.width) / 2.f);
		CGFloat yDelta = floorf((bounds.size.height - initialSize.height) / 2.f);
		broadcasterLabelFrame.origin.x += xDelta;
		broadcasterLabelFrame.origin.y += yDelta;
		broadcastTitleLabelFrame.origin.x += xDelta;
		broadcastTitleLabelFrame.origin.y += yDelta;
		viewerCountLabelFrame.origin.x += xDelta;
		viewerCountLabelFrame.origin.y += yDelta;
	}
	
	[UIView animateWithDuration:.25 animations:^{
		
		self.backgroundView.bounds = bounds;
		_channelNameLabel.frame = broadcasterLabelFrame;
		_broadcastTitleLabel.frame = broadcastTitleLabelFrame;
		_detailTextLabel.frame = viewerCountLabelFrame;
	}];
}

- (void)setHighlighted:(BOOL)highlighted
{
	[super setHighlighted:highlighted];
	[self adjustForHighlightState:highlighted];
}

- (void)layoutSubviews
{
	[super layoutSubviews];
	
	NSString *placeholderImageName = nil;
	if (CGSizeEqualToSize([[self class] portraitCellSize], self.frame.size))
		placeholderImageName = @"404_channel_ipad_grid";
	else placeholderImageName = @"404_sdk_ipad_landscape_directory";
	
	if (placeholderImageName) self.placeholderImageName = placeholderImageName;
	
	_enlargedStreamGradientOverlay.frame = CGRectMake(0.f, 0.f, self.frame.size.width, self.frame.size.height);
	
	if (CGSizeEqualToSize([[self class] portraitCellSize], self.frame.size)) {
		
		_enlargedStreamGradientOverlay.contents = (__bridge id)_portraitGradientOverlay.CGImage;
		
	}
	else if (CGSizeEqualToSize([[self class] landscapeCellSize], self.frame.size)) {
		
		_enlargedStreamGradientOverlay.contents = (__bridge id)_landscapeGradientOverlay.CGImage;
	}
}

#undef TWStreamCellTextPadding
#undef TWStreamCellBroadcasterLabelBottomOffset
#undef TWStreamCellBroadcasterLabelHeight
#undef TWStreamCellViewerCountLabelBottomOffset
#undef TWStreamCellViewerCountLabelHeight
	
@end
